<?php



/*

 * This file is part of the Symfony package.

 *

 * (c) Fabien Potencier <fabien@symfony.com>

 *

 * For the full copyright and license information, please view the LICENSE

 * file that was distributed with this source code.

 */



namespace Symfony\Component\Security\Core\User;



use Symfony\Component\Security\Core\Exception\AccountStatusException;

use Symfony\Component\Security\Core\Exception\AccountExpiredException;

use Symfony\Component\Security\Core\Exception\LockedException;

use Symfony\Component\Security\Core\Exception\CredentialsExpiredException;

use Symfony\Component\Security\Core\Exception\DisabledException;



/**

 * Adds extra features to a user class related to account status flags.

 *

 * This interface can be implemented in place of UserInterface if you'd like

 * the authentication system to consider different account status flags

 * during authentication. If any of the methods in this interface return

 * false, authentication will fail.

 *

 * If you need to perform custom logic for any of these situations, then

 * you will need to register an exception listener and watch for the specific

 * exception instances thrown in each case. All exceptions are a subclass

 * of AccountStatusException

 *

 * @see UserInterface

 * @see AccountStatusException

 *

 * @author Fabien Potencier <fabien@symfony.com>

 */

interface AdvancedUserInterface extends UserInterface

{

    /**

     * Checks whether the user's account has expired.

     *

     * Internally, if this method returns false, the authentication system

     * will throw an AccountExpiredException and prevent login.

     *

     * @return bool    true if the user's account is non expired, false otherwise

     *

     * @see AccountExpiredException

     */

    public function isAccountNonExpired();



    /**

     * Checks whether the user is locked.

     *

     * Internally, if this method returns false, the authentication system

     * will throw a LockedException and prevent login.

     *

     * @return bool    true if the user is not locked, false otherwise

     *

     * @see LockedException

     */

    public function isAccountNonLocked();



    /**

     * Checks whether the user's credentials (password) has expired.

     *

     * Internally, if this method returns false, the authentication system

     * will throw a CredentialsExpiredException and prevent login.

     *

     * @return bool    true if the user's credentials are non expired, false otherwise

     *

     * @see CredentialsExpiredException

     */

    public function isCredentialsNonExpired();



    /**

     * Checks whether the user is enabled.

     *

     * Internally, if this method returns false, the authentication system

     * will throw a DisabledException and prevent login.

     *

     * @return bool    true if the user is enabled, false otherwise

     *

     * @see DisabledException

     */

    public function isEnabled();

}

