<?php



/*

 * This file is part of the Symfony package.

 *

 * (c) Fabien Potencier <fabien@symfony.com>

 *

 * For the full copyright and license information, please view the LICENSE

 * file that was distributed with this source code.

 */



namespace Symfony\Component\HttpFoundation\Session\Storage\Handler;



/**

 * PdoSessionHandler.

 *

 * @author Fabien Potencier <fabien@symfony.com>

 * @author Michael Williams <michael.williams@funsational.com>

 * @author Tobias Schultze <http://tobion.de>

 */

class PdoSessionHandler implements \SessionHandlerInterface

{

    /**

     * @var \PDO PDO instance

     */

    private $pdo;



    /**

     * @var string Table name

     */

    private $table;



    /**

     * @var string Column for session id

     */

    private $idCol;



    /**

     * @var string Column for session data

     */

    private $dataCol;



    /**

     * @var string Column for timestamp

     */

    private $timeCol;



    /**

     * Constructor.

     *

     * List of available options:

     *  * db_table: The name of the table [required]

     *  * db_id_col: The column where to store the session id [default: sess_id]

     *  * db_data_col: The column where to store the session data [default: sess_data]

     *  * db_time_col: The column where to store the timestamp [default: sess_time]

     *

     * @param \PDO  $pdo       A \PDO instance

     * @param array $dbOptions An associative array of DB options

     *

     * @throws \InvalidArgumentException When "db_table" option is not provided

     */

    public function __construct(\PDO $pdo, array $dbOptions = array())

    {

        if (!array_key_exists('db_table', $dbOptions)) {

            throw new \InvalidArgumentException('You must provide the "db_table" option for a PdoSessionStorage.');

        }

        if (\PDO::ERRMODE_EXCEPTION !== $pdo->getAttribute(\PDO::ATTR_ERRMODE)) {

            throw new \InvalidArgumentException(sprintf('"%s" requires PDO error mode attribute be set to throw Exceptions (i.e. $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION))', __CLASS__));

        }

        $this->pdo = $pdo;

        $dbOptions = array_merge(array(

            'db_id_col'   => 'sess_id',

            'db_data_col' => 'sess_data',

            'db_time_col' => 'sess_time',

        ), $dbOptions);



        $this->table = $dbOptions['db_table'];

        $this->idCol = $dbOptions['db_id_col'];

        $this->dataCol = $dbOptions['db_data_col'];

        $this->timeCol = $dbOptions['db_time_col'];

    }



    /**

     * {@inheritdoc}

     */

    public function open($savePath, $sessionName)

    {

        return true;

    }



    /**

     * {@inheritdoc}

     */

    public function close()

    {

        return true;

    }



    /**

     * {@inheritdoc}

     */

    public function destroy($sessionId)

    {

        // delete the record associated with this id

        $sql = "DELETE FROM $this->table WHERE $this->idCol = :id";



        try {

            $stmt = $this->pdo->prepare($sql);

            $stmt->bindParam(':id', $sessionId, \PDO::PARAM_STR);

            $stmt->execute();

        } catch (\PDOException $e) {

            throw new \RuntimeException(sprintf('PDOException was thrown when trying to delete a session: %s', $e->getMessage()), 0, $e);

        }



        return true;

    }



    /**

     * {@inheritdoc}

     */

    public function gc($maxlifetime)

    {

        // delete the session records that have expired

        $sql = "DELETE FROM $this->table WHERE $this->timeCol < :time";



        try {

            $stmt = $this->pdo->prepare($sql);

            $stmt->bindValue(':time', time() - $maxlifetime, \PDO::PARAM_INT);

            $stmt->execute();

        } catch (\PDOException $e) {

            throw new \RuntimeException(sprintf('PDOException was thrown when trying to delete expired sessions: %s', $e->getMessage()), 0, $e);

        }



        return true;

    }



    /**

     * {@inheritdoc}

     */

    public function read($sessionId)

    {

        $sql = "SELECT $this->dataCol FROM $this->table WHERE $this->idCol = :id";



        try {

            $stmt = $this->pdo->prepare($sql);

            $stmt->bindParam(':id', $sessionId, \PDO::PARAM_STR);

            $stmt->execute();



            // We use fetchAll instead of fetchColumn to make sure the DB cursor gets closed

            $sessionRows = $stmt->fetchAll(\PDO::FETCH_NUM);



            if ($sessionRows) {

                return base64_decode($sessionRows[0][0]);

            }



            return '';

        } catch (\PDOException $e) {

            throw new \RuntimeException(sprintf('PDOException was thrown when trying to read the session data: %s', $e->getMessage()), 0, $e);

        }

    }



    /**

     * {@inheritdoc}

     */

    public function write($sessionId, $data)

    {

        // Session data can contain non binary safe characters so we need to encode it.

        $encoded = base64_encode($data);



        // We use a MERGE SQL query when supported by the database.

        // Otherwise we have to use a transactional DELETE followed by INSERT to prevent duplicate entries under high concurrency.



        try {

            $mergeSql = $this->getMergeSql();



            if (null !== $mergeSql) {

                $mergeStmt = $this->pdo->prepare($mergeSql);

                $mergeStmt->bindParam(':id', $sessionId, \PDO::PARAM_STR);

                $mergeStmt->bindParam(':data', $encoded, \PDO::PARAM_STR);

                $mergeStmt->bindValue(':time', time(), \PDO::PARAM_INT);

                $mergeStmt->execute();



                return true;

            }



            $this->pdo->beginTransaction();



            try {

                $deleteStmt = $this->pdo->prepare(

                    "DELETE FROM $this->table WHERE $this->idCol = :id"

                );

                $deleteStmt->bindParam(':id', $sessionId, \PDO::PARAM_STR);

                $deleteStmt->execute();



                $insertStmt = $this->pdo->prepare(

                    "INSERT INTO $this->table ($this->idCol, $this->dataCol, $this->timeCol) VALUES (:id, :data, :time)"

                );

                $insertStmt->bindParam(':id', $sessionId, \PDO::PARAM_STR);

                $insertStmt->bindParam(':data', $encoded, \PDO::PARAM_STR);

                $insertStmt->bindValue(':time', time(), \PDO::PARAM_INT);

                $insertStmt->execute();



                $this->pdo->commit();

            } catch (\PDOException $e) {

                $this->pdo->rollback();



                throw $e;

            }

        } catch (\PDOException $e) {

            throw new \RuntimeException(sprintf('PDOException was thrown when trying to write the session data: %s', $e->getMessage()), 0, $e);

        }



        return true;

    }



    /**

     * Returns a merge/upsert (i.e. insert or update) SQL query when supported by the database.

     *

     * @return string|null The SQL string or null when not supported

     */

    private function getMergeSql()

    {

        $driver = $this->pdo->getAttribute(\PDO::ATTR_DRIVER_NAME);



        switch ($driver) {

            case 'mysql':

                return "INSERT INTO $this->table ($this->idCol, $this->dataCol, $this->timeCol) VALUES (:id, :data, :time) " .

                    "ON DUPLICATE KEY UPDATE $this->dataCol = VALUES($this->dataCol), $this->timeCol = VALUES($this->timeCol)";

            case 'oci':

                // DUAL is Oracle specific dummy table

                return "MERGE INTO $this->table USING DUAL ON ($this->idCol = :id) " .

                    "WHEN NOT MATCHED THEN INSERT ($this->idCol, $this->dataCol, $this->timeCol) VALUES (:id, :data, :time) " .

                    "WHEN MATCHED THEN UPDATE SET $this->dataCol = :data";

            case 'sqlsrv':

                // MS SQL Server requires MERGE be terminated by semicolon

                return "MERGE INTO $this->table USING (SELECT 'x' AS dummy) AS src ON ($this->idCol = :id) " .

                    "WHEN NOT MATCHED THEN INSERT ($this->idCol, $this->dataCol, $this->timeCol) VALUES (:id, :data, :time) " .

                    "WHEN MATCHED THEN UPDATE SET $this->dataCol = :data;";

            case 'sqlite':

                return "INSERT OR REPLACE INTO $this->table ($this->idCol, $this->dataCol, $this->timeCol) VALUES (:id, :data, :time)";

        }

    }



    /**

     * Return a PDO instance

     *

     * @return \PDO

     */

    protected function getConnection()

    {

        return $this->pdo;

    }

}

