<?php

/*

 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS

 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT

 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR

 * A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT

 * OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,

 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT

 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,

 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY

 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT

 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE

 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

 *

 * This software consists of voluntary contributions made by many individuals

 * and is licensed under the MIT license. For more information, see

 * <http://www.doctrine-project.org>.

 */



namespace Doctrine\DBAL\Types;



use Doctrine\DBAL\Platforms\AbstractPlatform;



/**

 * DateTime type saving additional timezone information.

 *

 * Caution: Databases are not necessarily experts at storing timezone related

 * data of dates. First, of all the supported vendors only PostgreSQL and Oracle

 * support storing Timezone data. But those two don't save the actual timezone

 * attached to a DateTime instance (for example "Europe/Berlin" or "America/Montreal")

 * but the current offset of them related to UTC. That means depending on daylight saving times

 * or not you may get different offsets.

 *

 * This datatype makes only sense to use, if your application works with an offset, not

 * with an actual timezone that uses transitions. Otherwise your DateTime instance

 * attached with a timezone such as Europe/Berlin gets saved into the database with

 * the offset and re-created from persistence with only the offset, not the original timezone

 * attached.

 *

 * @link    www.doctrine-project.org

 * @since   1.0

 * @author  Benjamin Eberlei <kontakt@beberlei.de>

 * @author  Guilherme Blanco <guilhermeblanco@hotmail.com>

 * @author  Jonathan Wage <jonwage@gmail.com>

 * @author  Roman Borschel <roman@code-factory.org>

 */

class DateTimeTzType extends Type

{

    /**

     * {@inheritdoc}

     */

    public function getName()

    {

        return Type::DATETIMETZ;

    }



    /**

     * {@inheritdoc}

     */

    public function getSQLDeclaration(array $fieldDeclaration, AbstractPlatform $platform)

    {

        return $platform->getDateTimeTzTypeDeclarationSQL($fieldDeclaration);

    }



    /**

     * {@inheritdoc}

     */

    public function convertToDatabaseValue($value, AbstractPlatform $platform)

    {

        return ($value !== null)

            ? $value->format($platform->getDateTimeTzFormatString()) : null;

    }



    /**

     * {@inheritdoc}

     */

    public function convertToPHPValue($value, AbstractPlatform $platform)

    {

        if ($value === null || $value instanceof \DateTime) {

            return $value;

        }



        $val = \DateTime::createFromFormat($platform->getDateTimeTzFormatString(), $value);

        if ( ! $val) {

            throw ConversionException::conversionFailedFormat($value, $this->getName(), $platform->getDateTimeTzFormatString());

        }



        return $val;

    }

}

