<?php

/*

 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS

 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT

 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR

 * A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT

 * OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,

 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT

 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,

 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY

 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT

 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE

 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

 *

 * This software consists of voluntary contributions made by many individuals

 * and is licensed under the MIT license. For more information, see

 * <http://www.doctrine-project.org>.

 */



namespace Doctrine\DBAL\Platforms;



use Doctrine\DBAL\Schema\Sequence;



/**

 * Platform to ensure compatibility of Doctrine with Microsoft SQL Server 2012 version.

 *

 * Differences to SQL Server 2008 and before are that sequences are introduced.

 *

 * @author Steve Müller <st.mueller@dzh-online.de>

 */

class SQLServer2012Platform extends SQLServer2008Platform

{

    /**

     * {@inheritdoc}

     */

    public function getAlterSequenceSQL(Sequence $sequence)

    {

        return 'ALTER SEQUENCE ' . $sequence->getQuotedName($this) .

               ' INCREMENT BY ' . $sequence->getAllocationSize();

    }



    /**

     * {@inheritdoc}

     */

    public function getCreateSequenceSQL(Sequence $sequence)

    {

        return 'CREATE SEQUENCE ' . $sequence->getQuotedName($this) .

               ' START WITH ' . $sequence->getInitialValue() .

               ' INCREMENT BY ' . $sequence->getAllocationSize() .

               ' MINVALUE ' . $sequence->getInitialValue();

    }



    /**

     * {@inheritdoc}

     */

    public function getDropSequenceSQL($sequence)

    {

        if ($sequence instanceof Sequence) {

            $sequence = $sequence->getQuotedName($this);

        }



        return 'DROP SEQUENCE ' . $sequence;

    }



    /**

     * {@inheritdoc}

     */

    public function getListSequencesSQL($database)

    {

        return 'SELECT seq.name, seq.increment, seq.start_value FROM sys.sequences AS seq';

    }



    /**

     * {@inheritdoc}

     */

    public function getSequenceNextValSQL($sequenceName)

    {

        return 'SELECT NEXT VALUE FOR ' . $sequenceName;

    }



    /**

     * {@inheritdoc}

     */

    public function supportsSequences()

    {

        return true;

    }



    /**

     * {@inheritdoc}

     *

     * Returns Microsoft SQL Server 2012 specific keywords class

     */

    protected function getReservedKeywordsClass()

    {

        return 'Doctrine\DBAL\Platforms\Keywords\SQLServer2012Keywords';

    }

}

