<?php

/*

 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS

 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT

 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR

 * A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT

 * OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,

 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT

 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,

 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY

 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT

 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE

 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

 *

 * This software consists of voluntary contributions made by many individuals

 * and is licensed under the MIT license. For more information, see

 * <http://www.doctrine-project.org>.

 */



namespace Doctrine\Common\Persistence;



use Doctrine\Common\Persistence\ManagerRegistry;



/**

 * Abstract implementation of the ManagerRegistry contract.

 *

 * @link   www.doctrine-project.org

 * @since  2.2

 * @author Fabien Potencier <fabien@symfony.com>

 * @author Benjamin Eberlei <kontakt@beberlei.de>

 * @author Lukas Kahwe Smith <smith@pooteeweet.org>

 */

abstract class AbstractManagerRegistry implements ManagerRegistry

{

    /**

     * @var string

     */

    private $name;



    /**

     * @var array

     */

    private $connections;



    /**

     * @var array

     */

    private $managers;



    /**

     * @var string

     */

    private $defaultConnection;



    /**

     * @var string

     */

    private $defaultManager;



    /**

     * @var string

     */

    private $proxyInterfaceName;



    /**

     * Constructor.

     *

     * @param string $name

     * @param array  $connections

     * @param array  $managers

     * @param string $defaultConnection

     * @param string $defaultManager

     * @param string $proxyInterfaceName

     */

    public function __construct($name, array $connections, array $managers, $defaultConnection, $defaultManager, $proxyInterfaceName)

    {

        $this->name = $name;

        $this->connections = $connections;

        $this->managers = $managers;

        $this->defaultConnection = $defaultConnection;

        $this->defaultManager = $defaultManager;

        $this->proxyInterfaceName = $proxyInterfaceName;

    }



    /**

     * Fetches/creates the given services.

     *

     * A service in this context is connection or a manager instance.

     *

     * @param string $name The name of the service.

     *

     * @return object The instance of the given service.

     */

    abstract protected function getService($name);



    /**

     * Resets the given services.

     *

     * A service in this context is connection or a manager instance.

     *

     * @param string $name The name of the service.

     *

     * @return void

     */

    abstract protected function resetService($name);



    /**

     * Gets the name of the registry.

     *

     * @return string

     */

    public function getName()

    {

        return $this->name;

    }



    /**

     * {@inheritdoc}

     */

    public function getConnection($name = null)

    {

        if (null === $name) {

            $name = $this->defaultConnection;

        }



        if (!isset($this->connections[$name])) {

            throw new \InvalidArgumentException(sprintf('Doctrine %s Connection named "%s" does not exist.', $this->name, $name));

        }



        return $this->getService($this->connections[$name]);

    }



    /**

     * {@inheritdoc}

     */

    public function getConnectionNames()

    {

        return $this->connections;

    }



    /**

     * {@inheritdoc}

     */

    public function getConnections()

    {

        $connections = array();

        foreach ($this->connections as $name => $id) {

            $connections[$name] = $this->getService($id);

        }



        return $connections;

    }



    /**

     * {@inheritdoc}

     */

    public function getDefaultConnectionName()

    {

        return $this->defaultConnection;

    }



    /**

     * {@inheritdoc}

     */

    public function getDefaultManagerName()

    {

        return $this->defaultManager;

    }



    /**

     * {@inheritdoc}

     *

     * @throws \InvalidArgumentException

     */

    public function getManager($name = null)

    {

        if (null === $name) {

            $name = $this->defaultManager;

        }



        if (!isset($this->managers[$name])) {

            throw new \InvalidArgumentException(sprintf('Doctrine %s Manager named "%s" does not exist.', $this->name, $name));

        }



        return $this->getService($this->managers[$name]);

    }



    /**

     * {@inheritdoc}

     */

    public function getManagerForClass($class)

    {

        // Check for namespace alias

        if (strpos($class, ':') !== false) {

            list($namespaceAlias, $simpleClassName) = explode(':', $class);

            $class = $this->getAliasNamespace($namespaceAlias) . '\\' . $simpleClassName;

        }



        $proxyClass = new \ReflectionClass($class);

        if ($proxyClass->implementsInterface($this->proxyInterfaceName)) {

            $class = $proxyClass->getParentClass()->getName();

        }



        foreach ($this->managers as $id) {

            $manager = $this->getService($id);



            if (!$manager->getMetadataFactory()->isTransient($class)) {

                return $manager;

            }

        }

    }



    /**

     * {@inheritdoc}

     */

    public function getManagerNames()

    {

        return $this->managers;

    }



    /**

     * {@inheritdoc}

     */

    public function getManagers()

    {

        $dms = array();

        foreach ($this->managers as $name => $id) {

            $dms[$name] = $this->getService($id);

        }



        return $dms;

    }



    /**

     * {@inheritdoc}

     */

    public function getRepository($persistentObjectName, $persistentManagerName = null)

    {

        return $this->getManager($persistentManagerName)->getRepository($persistentObjectName);

    }



    /**

     * {@inheritdoc}

     */

    public function resetManager($name = null)

    {

        if (null === $name) {

            $name = $this->defaultManager;

        }



        if (!isset($this->managers[$name])) {

            throw new \InvalidArgumentException(sprintf('Doctrine %s Manager named "%s" does not exist.', $this->name, $name));

        }



        // force the creation of a new document manager

        // if the current one is closed

        $this->resetService($this->managers[$name]);

    }

}

